/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/


/* ---------------------------------------------------------------
 * File: xtypedef.h
 * -----------------------------------------------------------------*/

#ifndef XTYPEDEFS_H_INCLUDED
#define XTYPEDEFS_H_INCLUDED

#ifdef BEST_FIRMWARE
  do not include this file into firmware
#endif


#include <assert.h>
#include <stddef.h>
#include <stdio.h>
#ifndef __KERNEL__
#include <stdlib.h>
#include <string.h>
#include <time.h>
#else
#include <linux/time.h>
#include <linux/string.h>
#endif

/* this file contains basic defines (as type of OS or things like CLI
 * and IMPORT / EXPORT */
#include <xdefines.h>

#include <xberror.h>

#ifdef CUSTOM_OEM1
#include <xoemtype.h>
#endif

/* The following basic typedefs depend on the used machine. We distinguish
 * between HPUX Workstation and Intel based PCs with 16 or 32 Bit technology.

 * Because the Perl script convert.pl reads only lines starting with the
 * keyword CLI, it may read other typedefs than the preprocessor.
 * We dont have to worry about this because the generated Code
 * (dispatch.h) contains only the user defined typedefs, so that the
 * preprocessor can resolve them correctly.
 */

/* ---------------------------------------------------------------
 * General Type Definitions.
 * IMPORTANT; For all data streams or vectors you must use the
 * pointer types below.  This allows correct pointer arithmetic 
 * under DOS/16-bit.
 * --------------------------------------------------------------- */

#if defined(DOS) && !(defined(WIN32)||defined(WIN64)) /* 16-Bit PCs */
  CLI typedef unsigned char bx_int8;
  CLI typedef unsigned int bx_int16;
  CLI typedef unsigned long bx_int32;
  CLI typedef bx_int32 bx_int64;

  /* pointers to a data stream or array of... */
  CLI typedef bx_int8 HUGE * bx_int8ptr;
  CLI typedef bx_int16 HUGE * bx_int16ptr;
  CLI typedef bx_int32 HUGE * bx_int32ptr;
  CLI typedef bx_int64 HUGE * bx_int64ptr;

  /* pointers to const.. */
  CLI typedef const bx_int8 HUGE * bx_cint8ptr;
  CLI typedef const bx_int16 HUGE * bx_cint16ptr;
  CLI typedef const bx_int32 HUGE * bx_cint32ptr;
  CLI typedef const bx_int64 HUGE * bx_cint64ptr;

#elif defined(LINUX)

  #include <stdint.h>

  typedef unsigned long ULONG;

  typedef unsigned char bx_int8;
  typedef unsigned short bx_int16;
  typedef unsigned long bx_int32;
  typedef long long __int64;
  typedef uint_fast64_t bx_int64;
  typedef intptr_t bx_intptr;

  /* pointers to a data stream or array of... */
  typedef bx_int8 * bx_int8ptr;
  typedef bx_int16 * bx_int16ptr;
  typedef bx_int32 * bx_int32ptr;
  typedef bx_int64 * bx_int64ptr;

  /* pointers to const.. */
  typedef const bx_int8 * bx_cint8ptr;
  typedef const bx_int16 * bx_cint16ptr;
  typedef const bx_int32 * bx_cint32ptr;
  typedef const bx_int64 * bx_cint64ptr;

#else           /* 32-Bit PCs and HPUX */
  typedef unsigned char bx_int8;
  typedef unsigned short bx_int16;
  typedef unsigned long bx_int32;
  #ifdef UNIX
    typedef long __int64;
  #endif
  #ifdef WIN64
    typedef unsigned __int64 bx_int64;
    typedef unsigned __int64 bx_intptr;
  #else
    typedef unsigned long bx_int64;
    typedef unsigned long bx_intptr;
  #endif

  /* pointers to a data stream or array of... */
  typedef bx_int8 * bx_int8ptr;
  typedef bx_int16 * bx_int16ptr;
  typedef bx_int32 * bx_int32ptr;
  typedef bx_int64 * bx_int64ptr;

  /* pointers to const.. */
  typedef const bx_int8 * bx_cint8ptr;
  typedef const bx_int16 * bx_cint16ptr;
  typedef const bx_int32 * bx_cint32ptr;
  typedef const bx_int64 * bx_cint64ptr;

#endif


CLI typedef char bx_char;
CLI typedef char * bx_charptrtype;
CLI typedef const char * bx_ccharptrtype;
CLI typedef char ** bx_charptrptrtype;

/* constant pointers to.. */
CLI typedef const bx_charptrtype bx_charcptrtype;
CLI typedef const bx_int8ptr bx_int8cptr;
CLI typedef const bx_int16ptr bx_int16cptr;
CLI typedef const bx_int32ptr bx_int32cptr;
CLI typedef const bx_int64ptr bx_int64cptr;

CLI typedef int bx_bool;
CLI typedef unsigned char bx_sizetype;


/* replaced x with a and y with b */
#ifndef __cplusplus
#  ifndef max
#    define max(a,b)    (((a) > (b)) ? (a) : (b))
#  endif
#  ifndef min
#    define min(a,b)    (((a) < (b)) ? (a) : (b))
#  endif
#endif


/* The numer of cards, which can be opened simulaneously */
#define BX_MAXHANDLES    256


/* --------------------------------------------------------------------------
 * BX_ERRETURN() is used for ALL returns from exported functions.
 * BX_ERRCHECK() is the standard call of an exported CAPI function.
 *
 * NOTE; These macros MUST NOT EVALUATE ErrValOrFuncReturn more than once !!!
 * -------------------------------------------------------------------------- */

#define BX_ERRETURN(ErrValOrFuncReturn) \
  return BestXLastErrorSet(handle, (ErrValOrFuncReturn))

#define BX_ERRCHECK(ErrValOrFuncReturn)  \
  if (BX_E_OK != (__status= ErrValOrFuncReturn)) { BX_ERRETURN(__status); }


/* --------------------------------------------------------------------------
 * BX_LICENSECHECK and BX_TRY_LICENSE have the same functionality. 
 * BX_TRY_LICENSE is more efficient if you use a try block.
 * -------------------------------------------------------------------------- */

#define __BX_HANDLECHECK    ((handle >= 0) &&             \
             (handle < BX_MAXHANDLES) &&     \
             bx_handlearray[handle].is_open)

#define BX_HANDLECHECK if (!__BX_HANDLECHECK) { BX_ERRETURN (BX_E_BAD_HANDLE); }

#define BX_LICENSECHECK(item) { BX_HANDLECHECK \
                               BX_ERRCHECK(BestXCapabilityCheck(handle, (item)));}

#define BX_TRY_HANDLE  if (!__BX_HANDLECHECK) { BX_TRY_ERROR (BX_E_BAD_HANDLE); }

#define BX_TRY_LICENSE(item)  { BX_TRY_HANDLE; \
                               BX_TRY (BestXCapabilityCheck(handle, (item))); }



/* --------------------------------------------------------------------------
 * Function parameter checking macros.
 * NOTE; Every function using one of these macros MUST have its name 
 * declared at the beginning of the function thus;
 *
 * BX_DECLARE_FUNCNAME("BestXMyFunctionName [cliname]")
 * BX_FCT_PARAM_CHK(2, pin_num > 7);
 * // if pin_num (param number 2) really is > 7 then a msg is displayed
 * -------------------------------------------------------------------------- */

#define BX_DECLARE_FUNCNAME(funcname) \
    static bx_ccharptrtype pFuncName = (funcname)

#define BX_FCT_PARAM_MSG(param, reason)             \
        BestXFuncParamMsg(handle, pFuncName, #param, reason)

/* TODO: add value of parameter into makro */

/* Use these normally */

#define BX_FCT_PARAM_CHK(param, boolean_test)   \
                if ((boolean_test)) {    \
                  BX_FCT_PARAM_MSG ((param), (#boolean_test)); \
                  BX_ERRETURN (BX_E_FCT_PARAM);   \
                }

#define BX_FCT_PARAM_CHK_R(param, boolean_test, reason) \
                if ((boolean_test)) {   \
                  BX_FCT_PARAM_MSG ((param), (reason)); \
                  BX_ERRETURN (BX_E_FCT_PARAM);  \
                }

#define BX_FCT_PARAM_ERROR(param, reason)  \
                {     \
                  BX_FCT_PARAM_MSG ((param), (reason)); \
                  BX_ERRETURN (BX_E_FCT_PARAM);  \
                }

#define BX_FCT_PARAM_NULL_POINTER_CHK(param)    \
        BX_FCT_PARAM_CHK_R(param, param == NULL, "null pointer")

#define BX_FCT_PARAM_ALIGNMENT_CHK(param, alignment)  \
        {       \
          char    s2 [32];     \
        \
          sprintf (s2, "must be %d-aligned", alignment); \
          BX_FCT_PARAM_CHK_R(param, (param%alignment), s2); \
        }

#define BX_FCT_PARAM_RANGE_CHK(param, lobound, hibound) \
        BX_FCT_PARAM_CHK(param, !(param >= lobound)); \
        BX_FCT_PARAM_CHK(param, !(param <= hibound)) \


/* Use these within try blocks */

#define BX_TRY_FCT_PARAM(param, boolean_test)           \
                if ((boolean_test))                             \
                {                                                 \
                  BX_FCT_PARAM_MSG ((param), (#boolean_test)); \
                  BX_TRY_ERROR (BX_E_FCT_PARAM);                  \
                }

#define BX_TRY_FCT_PARAM_R(param, boolean_test, reason) \
                if ((boolean_test)) {   \
                  BX_FCT_PARAM_MSG ((param), (reason)); \
                  BX_TRY_ERROR (BX_E_FCT_PARAM); \
                }     \

#define BX_TRY_FCT_PARAM_ERROR(param, reason)  \
                {     \
                  BX_FCT_PARAM_MSG ((param), (reason)); \
                  BX_TRY_ERROR (BX_E_FCT_PARAM); \
                }

#define BX_TRY_FCT_PARAM_NULL_POINTER(param)        \
        BX_TRY_FCT_PARAM_R(param, param == NULL, "null pointer")

#define BX_TRY_FCT_PARAM_ALIGNMENT(param, alignment)                \
        {                                \
          char    s3 [32];                        \
                                        \
          sprintf (s3, "must be %d-aligned", alignment);        \
          BX_TRY_FCT_PARAM_R(param,(param%alignment), s3);  \
        }

/* TODO: make look similar to property checks */
#define BX_TRY_FCT_PARAM_RANGE(param, lobound, hibound)    \
        BX_TRY_FCT_PARAM(param, !(param >= lobound));        \
        BX_TRY_FCT_PARAM(param, !(param <= hibound))

                           
/* --------------------------------------------------------------------------
   TRY blocks:

   Try blocks are an efficient way of catching errors in a series of 
   CAPI function calls and are particularly useful for situations where 
   some cleanup is required after an error occurs.  
   In essence this is a type of "goto" mechanism that allows every error 
   condition to follow the same path out of the program.  
   
   Processing within the try block will stop at the first error.
   
   *** All functions in TRY macros must return bx_errtype. ***

   How to use (example):

   BX_TRY_VARS_NO_PROG;            # declares all necessary variables

   BX_TRY_BEGIN                     # starts a try block
   {
     BX_TRY (MyFunction1 (...));        # call functions returning bx_errtype
     BX_TRY (MyFunction2 (...));

     ... more TRY statements        # other calls can be included

     BX_TRY_ERROR (BX_E_SOME_ERROR)       # fail with own error code

     BX_TRY_FAIL (test ? BX_E_OK : BX_E_SOME_ERROR); 
                                        # test and fail with own error code 
                         # block will continue if the error
                                        # expression evaluates to BX_E_OK
   }

   BX_TRY_CATCH                    # the catch block is optional
   {
     ... free memory, perform common cleanup,
     ... add special error messages etc. here
   }

   BX_ERRETURN(BX_TRY_RET);        # the reason for the failure
                       # you can evaluate this makro
                                        # wherever you like.


   ===========================================================================
   For situations where the cleanup depends on which instructions or calls
   were made before the error was encountered there is a "progress" mechanism
   for tracking this.  Here is an example of using progress;

   BX_TRY_VARS;

   BX_TRY_BEGIN
   {
     ...call a normal CAPI function
     BX_TRY (SomeFunction (...));

     ...start something
     BX_TRY_PROGRESS ( StartSomething() );

     ... more TRY statements

   }

   BX_TRY_CATCH                    # the catch block is NOT optional
   {
     BX_TRY_PASSED
     {
        ... we got an error AFTER something was started
        (void) StopSomething();
     }
     BX_TRY_FAILED
     {
        ... something was never started...
     }
   }

   BX_ERRETURN(BX_TRY_RET);

   NOTE; The use of progress tracking makes a function difficult to read
   and maintain and should be limited to cases where it's really necessary.
 * -------------------------------------------------------------------------- */


#include <setjmp.h>

#define BX_TRY_VARS_NO_PROG      jmp_buf     __env;              \
                                volatile bx_errtype __status = BX_E_OK; \
                              volatile __btmp

#define BX_TRY_VARS        jmp_buf     __env;              \
                      volatile bx_errtype __status = BX_E_OK;  \
                      volatile __btmp; \
                      volatile int __progress = 0


/* --------------------------------------------------------------------------
 * DO NOT (***NOT***) make ANY of these macros more than ONE (1) statement.
 * They are used in hundreds of places and are not always enclosed in braces
 * -------------------------------------------------------------------------- */

#define BX_TRY_BEGIN   __btmp = setjmp(__env); if (__btmp == 0)
#define BX_TRY(fct)    if (BX_E_OK != (__status = (fct))) longjmp(__env, 1)
#define BX_TRY_PROGRESS(fct)    BX_TRY ((fct)); ++__progress
#define BX_TRY_FAIL(reason)    BX_TRY ((reason))
#define BX_TRY_ERROR(errorcode)  __status = (errorcode); longjmp(__env, 1)
#define BX_TRY_CATCH        else
#define BX_TRY_PASSED        if (__progress-- <= 0 ? 0 : 1)
#define BX_TRY_FAILED        else
#define BX_TRY_RESET        __progress = 0
#define BX_TRY_RET     __status


/* ************************************************************************ */



/* The handle represents the number, under which the user can reference his
 * session. There is a struct behind this handle that holds all session
 * related information. */
CLI typedef int bx_handletype;

/* We need an Invalid Handle for the BestX handle array AND an invalid handle for the OS */
#define BX_INVALID_HANDLE BX_MAXHANDLES


/* this is the invalid OS handle (used by Windows and cast into bx_portnumtype) */
#if defined(WIN32) || defined(WIN64)

/* we use the WINDOWS definition for INVALID_HANDLE_VALUE to mark an invalid OS handle */
#define INVALID_OS_HANDLE INVALID_HANDLE_VALUE
typedef void *bx_portnumtype;

#else

#define INVALID_OS_HANDLE -1
typedef long int bx_portnumtype;

#endif


/* ---------------------------------------------------------------
 * Type Definitions for Session Functions
 * --------------------------------------------------------------- */
enum
{
  BX_PORT_COM1 = 1,
  BX_PORT_COM2 = 2,
  BX_PORT_COM3 = 3,
  BX_PORT_COM4 = 4
};

enum
{
  BX_PORT_LPT1 = 1,
  BX_PORT_LPT2 = 2
};


/* --------------------------------------------------------------------------
 * Timeouts.  For consistency we use the Win32 COMMTIMEOUTS timeout structure.
 * It is very flexible... redefined here to allow use by non-Win32 platforms.
 * Each handle has storage for one timeout structure but this is not used
 * if the device driver has its own storage...(i.e. Kernel-mode drivers).
 * This structure is also declared in timeouts.h for use by device drivers.
 * -------------------------------------------------------------------------- */

#if !BESTTIMEOUTS_DECL
#define BESTTIMEOUTS_DECL 1
typedef struct _BESTTIMEOUTS
{
  bx_int32 ReadIntervalTimeout;  /* Maximum time between read chars. */
  bx_int32 ReadTotalTimeoutMultiplier; /* Multiplier of characters.        */
  bx_int32 ReadTotalTimeoutConstant; /* Constant in milliseconds.        */
  bx_int32 WriteTotalTimeoutMultiplier;  /* Multiplier of characters.        */
  bx_int32 WriteTotalTimeoutConstant;  /* Constant in milliseconds.        */

} BESTTIMEOUTS, *LPBESTTIMEOUTS;
#endif          /* !BESTTIMEOUTS_DECL */


/* --------------------------------------------------------------------------
 * Hardware options.  The bx_hwinfotype structure is part of the "handle"
 * and is filled in during open by BestXIdHardware() (static in session.c).
 * It should be considered read-only from that point on!
 * -------------------------------------------------------------------------- */

/* the boardtypes we support ... also used as index into the global lookuptable
 */
typedef enum
{
  BX_HW_E2929A  = 0,
  BX_HW_E2929A_DEEP,
  BX_HW_E2922A,
  BX_HW_E2929B,
  BX_HW_E2929B_DEEP,
  BX_HW_E2922B,
  BX_HW_E2930A, /* FPGA */
  BX_HW_E2923A,
  BX_HW_E2930B,
  BX_HW_E2923B,
  BX_HW_E2923F, /* FPGA */
  BX_HW_UNKNOWN = 255   /* must be last entry! */
} bx_hwtype;


/* --------------------------------------------------------------------------
 * The board SERIES ... basically the same as bx_hwtype except no _DEEP
 * ------------------------------------------------------------------------- */
typedef enum
{
  BX_SERIES_E2929A = 0,
  BX_SERIES_E2922A,
  BX_SERIES_E2929B,
  BX_SERIES_E2922B,
  BX_SERIES_E2930A,
  BX_SERIES_E2923A,
  BX_SERIES_E2930B,
  BX_SERIES_E2923B,
  BX_SERIES_E2923F,
  BX_SERIES_UNKNOWN = 255
} bx_hwseriestype;


/* --------------------------------------------------------------------------
 * Encapsulation of all static hardware info about the currently open board.
 * DO NOT CHANGE THIS STRUCTURE WITHOUT corresponding changes in the
 * initialization array (see BestXIdHardware() in session.c).
 * *** DO NOT directly read this structure outside of session.c ***
 * -------------------------------------------------------------------------- */

#ifndef SWIG
typedef struct
{
  char *product_string;         /* essential for connecting to a card */
  bx_hwtype hw;                  /* compatibility...old bx_handlearray[].hw */
  bx_hwseriestype hwseries;      /* hardware series (no deep id) */
/*  bx_regfiletype regfile;  compatibility...old bx_handlearray[].regfile */
  bx_int32 dev_id;               /* pci device id stored in card's config space */
  bx_int32 hwbitmask;            /* access funcs in session.c */
} bx_hwinfotype;
#endif

#define BX_HW_INFO_UNKNOWN {"UNKNOWN", BX_HW_UNKNOWN, BX_SERIES_UNKNOWN, 0, 0}



CLI                             /* DEF_START ~G_BX_General */
/* ---------------------------------------------------------------
 * General Defines
 * --------------------------------------------------------------- */
#define BX_SIZE_BYTE   1         /* @byte  */
#define BX_SIZE_WORD   2         /* @word  */
#define BX_SIZE_DWORD  4         /* @dword */
CLI                             /* DEF_END */

CLI                             /* DEF_START ~G_BX_Baudrate */
/* ---------------------------------------------------------------
 * Defines for Session Functions
 * --------------------------------------------------------------- */
#define BX_BD_2400   2400
#define BX_BD_4800   4800
#define BX_BD_9600   9600
#define BX_BD_19200  19200
#define BX_BD_38400  38400
#define BX_BD_57600  57600
CLI                             /* DEF_END */


/* ---------------------------------------------------------------
 * Defines for Interrupt Generation Functions
 * --------------------------------------------------------------- */
CLI                             /* DEF_START ~G_BX_G19 */
#define BX_INTA             0x1  /* @IntA */
#define BX_INTB             0x2  /* @IntB */
#define BX_INTC             0x4  /* @IntC */
#define BX_INTD             0x8  /* @IntD */
CLI                             /* DEF_END */

CLI    /* DEF_START ~G_yesno */
#ifndef BX_NO
#  define BX_NO      0x0UL  /* @No */
#endif
#ifndef BX_YES
#  define BX_YES     0x1UL  /* @Yes */
#endif
#ifndef BX_FALSE
#  define BX_FALSE     BX_NO  /* @False */
#endif
#ifndef BX_TRUE
#  define BX_TRUE     BX_YES /* @True */
#endif
CLI                             /* DEF_END */

CLI    /* DEF_START ~G_onoff */
#define BX_OFF      0x0UL  /* @Off */
#define BX_ON      0x1UL  /* @On */
CLI                             /* DEF_END */

/* ---------------------------------------------------------------
 * Defines for Miscellaneous Functions
 * --------------------------------------------------------------- */


#include <xfirm.h>

/*----------------------------------------------------------------
 * the struct that defines the handle and what the content is
 *----------------------------------------------------------------*/

/* Note: this struct contains all information that belongs to a handle
 * in case this is changed, change also the static initialisation in
 * the file session.c

 * Important; leave "port" as the first element.
 * It is used in all the port switches and thus speed is important.

 * NOTE; the meaning of entered_port is dependent on bx_porttype...
 *       and will change if BestXDevIdentifierGet() or BestXHIFCardIdentifierGet()
 *       is used (i.e. with PCI or HIF).
 */

typedef struct bx_dynamic_capability_type bx_dynamic_capability_type; 
typedef struct bx_dbtype bx_dbtype; 

/* the opentask defines what type of open we support */
typedef enum
{
  BX_OPEN_STANDARD,
  BX_OPEN_STANDARD_RESERVED,
  BX_OPEN_FAILSAFE,
  BX_OPEN_LICENSE,
  BX_OPEN_BIOS_SWITCH,
  BX_OPEN_QUICKCHECK,
  BX_OPEN_RESERVE,
  BX_OPEN_USB23,
  BX_OPEN_FAST
} bx_OpenType;


typedef enum
{
  BX_CLOSE_STANDARD,
  BX_CLOSE_RESERVED,
  BX_CLOSE_USB23
} bx_closetype;



#ifndef SWIG
typedef struct
{
  bx_porttype port;           /* enum bx_porttype ... BX_PORT_PCI_CONF etc. */
  bx_portnumtype portnumber;  /* OS internal ref. number for port accesses */
  bx_int32 entered_port;         /* symbolic port number as entered
                                 * (BX_PORT_COM1 etc. ) */
  bx_int8 is_open;
  bx_int8 is_connected;
  bx_int8 is_reserved;           
  bx_hwinfotype hwinfo;         
  bx_int32 param;
  bx_int8 regwidth;           
  BESTTIMEOUTS timeouts;       
#ifndef SWIG
  void (*cbx_printf) (char *,...);     /* these functions are callback */
  void (*cbx_p_backup) (char *,...);   /* functions that get called    */
  void (*cbx_setprompt) (char *);      /* in debugging and command line */
  int (*cbx_getevent) (void);          /* cases */
#endif
  bx_dynamic_capability_type *capable; /* dynamic capabilities */
  bx_int32 perfboard_gapmode;
  bx_lasterrtype lasterr;              /* store information to build 
                                          a good error string */
  bx_dbtype *db;                       /* Pointer to setting database */

  /* CardIsInProgrammingMode is used to keep the actual number of switchings
     betweeen Run- and Prog- mode as small as possible.
     It is incremented for every call to BestXExerciserProgMode()
     and decremented in BestXExerciserRunMode().
     When it is decremented to zero, we actually switch 
     the card into Run-mode. 
     When it is incremented to one, we actually switch 
     the card into Programming-mode.
     Calls to BestXExerciserProgMode() and BestXExerciserRunMode()
     should usually be balanced !
     After a power cycle, the card starts up in Prog-mode.
  */
  
  bx_int32 CardIsInProgrammingMode;    /* =0, if card is in RunMode  */

  bx_int32 m_bufferSize;
  bx_int8ptr m_pBuffer; /* general purpose char buffer, */
                        /* used for blockregwrite accesses */

  bx_int32 FaustLoadError; /* 1, if FaustFPGA not loaded */
  bx_OpenType OpenType;


#if defined(_AIX) || defined(AIX)
  void *AIX_handle;         /* will be pointer to TU_GLOBAL_DATA */
#ifdef ADD_OEM_DATA
  OEMDataType  OEMData ;
#endif
#endif

} bx_handlestype;
#endif

#ifndef SWIG
extern bx_handlestype bx_handlearray[];

#endif
#if defined(AIX) || defined(_AIX)
 extern bx_handlestype xhandle_null;
#endif


/* the following are needed for BestXPortScan stuff */

#define BX_SIZEOF_POP_STR  24  /* in bytes */

#ifndef SWIG
/* Structure to retrieve information about a port */
typedef struct _BX_DEVICE_DATA
{
  const char * product_string   ;      /* "E2929A" etc. */
  char serial_number[21];        /* needed for unique card ID */
  char port_string[BX_SIZEOF_POP_STR]; /* "COM 1", "PCI Bus 0 Slot 7" etc. */
  bx_porttype port;                    /* BX_PORT_RS232 etc. */
  unsigned long portnum;               /* 0, 1, 0x1234 etc. */
  bx_hwinfotype hwinfo;                /* the rest of the story ... be careful! */

} BX_DEVICE_DATA;
#endif
        
#endif
